"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const app_1 = require("./src/v1/app");
const app_config_1 = require("./src/v1/configs/app.config");
const logger_1 = require("./src/v1/loggers/logger");
const datasource_plugin_1 = require("./src/v1/plugins/datasource.plugin");
const { port, host, environment } = app_config_1.AppConfig;
/**
 * * Export the Express app directly for Vercel to handle
 */
exports.default = app_1.App;
/**
 * * create express server with port and host imported from app.config
 * * This will only run when deployed in environments where Vercel is not handling the server (local environments).
 */
if (process.env.NODE_ENV !== "production") {
    app_1.App.listen(Number(port), host, () => {
        logger_1.Logger.debug("Express is running on →");
        console.table({
            host: host,
            port: port,
            environment: environment,
        });
    });
}
/**
 * * this method is for gracefully closing the express server(node.js process)
 * @function graceFullyCloseServerAndPluginConnections(exitCode)
 * * this function will first close the http server and then close MongoDB and Redis plugin connections
 * * and then proceed with process.exit(exitCode)
 */
const graceFullyCloseServerAndPluginConnections = (exitCode) => {
    const server = app_1.App.listen();
    server.close(() => {
        logger_1.Logger.debug("Closing the Server...");
        (0, datasource_plugin_1.DestroyRwwsDataSourcePluginConnection)();
        logger_1.Logger.debug(`Closing the main process with exitCode: ${exitCode}`);
        process.exit(exitCode);
    });
};
/**
 * * Handle various process events for graceful shutdown
 */
[`SIGINT`, `SIGUSR1`, `SIGUSR2`, `SIGTERM`].forEach((event) => {
    process.on(event, () => {
        logger_1.Logger.debug("Process event type: %s", event);
        graceFullyCloseServerAndPluginConnections(0);
    });
});
process.on("uncaughtException", (error) => {
    logger_1.Logger.error("uncaughtException-error:", error);
    process.exit(1);
});
process.on("unhandledRejection", (reason, promise) => {
    logger_1.Logger.error("unhandledRejection-at %s, %s", promise, `reason: ${reason}`);
    process.exit(1);
});
process.on("beforeExit", (code) => {
    logger_1.Logger.debug(`Process beforeExit event with code: ${code}`);
});
process.on("exit", (code) => {
    logger_1.Logger.debug(`Process exit event with code: ${code}`);
});
//# sourceMappingURL=index.js.map