"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.newsController = void 0;
const badRequest_error_1 = require("../errors/badRequest.error");
const http_response_1 = require("../responses/http.response");
const news_dao_1 = require("../dao/news.dao");
const news_utility_1 = require("../utilities/news.utility");
const path_1 = __importDefault(require("path"));
const notFound_error_1 = require("../errors/notFound.error");
const news_repository_1 = require("../repositories/news.repository");
const logger_1 = require("../loggers/logger");
const news_dto_1 = require("../dto/news.dto");
const newsDao = new news_dao_1.NewsDAO();
const createOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const createRequest = res.locals.reqBody;
        if (!req.file) {
            throw new badRequest_error_1.BadRequestError("image-not-present", "Please provide required news image", 3013);
        }
        createRequest.image = path_1.default.join("/uploads/", req.file.filename);
        createRequest.current_user = res.locals.user;
        const newsEntityObj = yield (0, news_utility_1.generateNewsEntityObject)(createRequest);
        const createdNews = yield newsDao.save(newsEntityObj);
        const newsDTO = new news_dto_1.NewsDTO(createdNews);
        return (0, http_response_1.Success)(res, {
            message: "Successfully created news",
            data: newsDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "createOne-news-base-error";
        error.message = error.message ? error.message : "createOne-news-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const getAll = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { page, pageSize, categoryId, year } = req.query;
        logger_1.Logger.debug("query-param %s", req.query);
        if (!page || !pageSize || (pageSize && pageSize > 100)) {
            throw new notFound_error_1.NotFoundError("news-search-pageSize-more-than-100", "news search page or page size not given or page size is more than 100", 3087);
        }
        const queryBuilder = news_repository_1.NewsRepository.createQueryBuilder("news")
            .innerJoinAndSelect("news.category", "category")
            .innerJoinAndSelect("news.createdBy", "createdBy");
        if (categoryId) {
            queryBuilder.where("category.id = :categoryId", { categoryId });
        }
        if (year) {
            queryBuilder.andWhere("EXTRACT(YEAR from news.created_at) = :year", { year });
        }
        queryBuilder.orderBy("news.updated_at", "DESC");
        const pageNumber = page || 1;
        const itemsPerPage = pageSize || 10;
        const skip = (pageNumber - 1) * itemsPerPage;
        const totalCount = yield queryBuilder.getCount();
        queryBuilder.skip(skip).take(pageSize);
        const newsData = yield queryBuilder.getMany();
        const newsList = newsData.map(news => new news_dto_1.NewsDTO(news));
        return (0, http_response_1.Success)(res, {
            message: "News fetched",
            data: {
                newsList,
                total_count: totalCount,
            },
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getAll-news-base-error";
        error.message = error.message ? error.message : "getAll-news-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const getOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { newsId } = req.params;
        const fetchedNews = yield newsDao.findOneById(newsId);
        if (!fetchedNews) {
            throw new notFound_error_1.NotFoundError("getOne-no-news-with-provided-id", "No news found or the news id is invalid", 3015);
        }
        const newsDTO = new news_dto_1.NewsDTO(fetchedNews);
        return (0, http_response_1.Success)(res, {
            message: "News detail fetched",
            data: newsDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getOne-news-base-error";
        error.message = error.message ? error.message : "getOne-news-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const updateOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const updateNewsRequestBody = res.locals.reqBody;
        const { newsId } = req.params;
        const targetNews = yield newsDao.findOneById(newsId);
        if (!targetNews) {
            throw new notFound_error_1.NotFoundError("updateOne-no-news-with-provided-id", "No news found or the news id is invalid", 3016);
        }
        req.file
            ? (updateNewsRequestBody.image = path_1.default.join("/uploads/", req.file.filename))
            : (updateNewsRequestBody.image = targetNews.image);
        const updatedNewsEntityObj = yield (0, news_utility_1.generateUpdatedNewsEntityObject)(updateNewsRequestBody, targetNews);
        const updatedNews = yield newsDao.save(updatedNewsEntityObj);
        const newsDTO = new news_dto_1.NewsDTO(updatedNews);
        return (0, http_response_1.Success)(res, {
            message: "News details updated",
            data: newsDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getOne-news-base-error";
        error.message = error.message ? error.message : "getOne-news-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const deleteOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { newsId } = req.params;
        const fetchedNews = yield newsDao.findOneById(newsId);
        if (!fetchedNews) {
            throw new notFound_error_1.NotFoundError("deleteOne-no-news-with-provided-id", "No news found or the news id is invalid", 3016);
        }
        yield news_repository_1.NewsRepository.remove(fetchedNews);
        return (0, http_response_1.Success)(res, {
            message: "News removed successfully",
            data: null,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "deleteOne-news-base-error";
        error.message = error.message ? error.message : "deleteOne-news-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
exports.newsController = {
    getAll,
    createOne,
    updateOne,
    getOne,
    deleteOne,
};
//# sourceMappingURL=news.controller.js.map