"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.programController = void 0;
const badRequest_error_1 = require("../errors/badRequest.error");
const http_response_1 = require("../responses/http.response");
const path_1 = __importDefault(require("path"));
const notFound_error_1 = require("../errors/notFound.error");
const program_dao_1 = require("../dao/program.dao");
const program_utility_1 = require("../utilities/program.utility");
const program_dto_1 = require("../dto/program.dto");
const program_repository_1 = require("../repositories/program.repository");
const programGoal_repository_1 = require("../repositories/programGoal.repository");
const programLocation_repository_1 = require("../repositories/programLocation.repository");
const programAchievement_repository_1 = require("../repositories/programAchievement.repository");
const logger_1 = require("../loggers/logger");
const programDao = new program_dao_1.ProgramDAO();
const createOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const createRequest = res.locals.reqBody;
        if (!req.file) {
            throw new badRequest_error_1.BadRequestError("image-not-present", "Please provide required program image", 3013);
        }
        createRequest.image = path_1.default.join("/uploads/", req.file.filename);
        const programEntityObj = yield (0, program_utility_1.generateProgramEntityObject)(createRequest);
        const createdProgram = yield programDao.save(programEntityObj);
        const programDTO = new program_dto_1.ProgramDTO(createdProgram);
        return (0, http_response_1.Success)(res, {
            message: "Successfully created program",
            data: programDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "createOne-program-base-error";
        error.message = error.message ? error.message : "createOne-program-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const getAll = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { page, pageSize, categoryId, status } = req.query;
        logger_1.Logger.debug("query-param %s", req.query);
        if (!page || !pageSize || (pageSize && pageSize > 100)) {
            throw new notFound_error_1.NotFoundError("program-search-pageSize-more-than-100", "program search page or page size not given or page size is more than 100", 3087);
        }
        const queryBuilder = program_repository_1.ProgramRepository.createQueryBuilder("program")
            .innerJoinAndSelect("program.category", "category")
            .leftJoinAndSelect("program.locations", "locations")
            .leftJoinAndSelect("program.goals", "goals")
            .leftJoinAndSelect("program.achievements", "achievements");
        if (categoryId) {
            queryBuilder.where("`category`.id = :categoryId", { categoryId });
        }
        if (status) {
            queryBuilder.andWhere("program.status = :status", { status });
        }
        queryBuilder.orderBy("program.updated_at", "DESC");
        const pageNumber = page || 1;
        const itemsPerPage = pageSize || 10;
        const skip = (pageNumber - 1) * itemsPerPage;
        const totalCount = yield queryBuilder.getCount();
        queryBuilder.skip(skip).take(pageSize);
        const programData = yield queryBuilder.getMany();
        const programList = programData.map(program => new program_dto_1.ProgramDTO(program));
        return (0, http_response_1.Success)(res, {
            message: "Programs fetched",
            data: {
                programList: programList,
                total_count: totalCount,
            },
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getAll-program-base-error";
        error.message = error.message ? error.message : "getAll-program-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const getOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { programId } = req.params;
        const fetchedProgram = yield programDao.findOneById(programId);
        if (!fetchedProgram) {
            throw new notFound_error_1.NotFoundError("getOne-no-program-with-provided-id", "No program found or the program id is invalid", 3015);
        }
        const programDTO = new program_dto_1.ProgramDTO(fetchedProgram);
        return (0, http_response_1.Success)(res, {
            message: "Program details fetched",
            data: programDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getOne-program-base-error";
        error.message = error.message ? error.message : "getOne-program-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const updateOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const updateProgramRequestBody = res.locals.reqBody;
        const { programId } = req.params;
        const targetProgram = yield programDao.findOneById(programId);
        if (!targetProgram) {
            throw new notFound_error_1.NotFoundError("updateOne-no-program-with-provided-id", "No program found or the program id is invalid", 3016);
        }
        req.file
            ? (updateProgramRequestBody.image = path_1.default.join("/uploads/", req.file.filename))
            : (updateProgramRequestBody.image = targetProgram.image);
        const updatedProgramEntityObj = yield (0, program_utility_1.generateUpdateProgramEntityObject)(updateProgramRequestBody, targetProgram);
        const updatedProgram = yield programDao.save(updatedProgramEntityObj);
        const programDTO = new program_dto_1.ProgramDTO(updatedProgram);
        return (0, http_response_1.Success)(res, {
            message: "Program details updated",
            data: programDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getOne-program-base-error";
        error.message = error.message ? error.message : "getOne-program-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const deleteOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { programId } = req.params;
        const fetchedProgram = yield programDao.findOneById(programId);
        if (!fetchedProgram) {
            throw new notFound_error_1.NotFoundError("deleteOne-no-program-with-provided-id", "No program found or the program id is invalid", 3016);
        }
        if (fetchedProgram.goals && fetchedProgram.goals.length > 0) {
            yield programGoal_repository_1.ProgramGoalRepository.remove(fetchedProgram.goals);
        }
        if (fetchedProgram.locations && fetchedProgram.locations.length > 0) {
            yield programLocation_repository_1.ProgramLocationRepository.remove(fetchedProgram.locations);
        }
        if (fetchedProgram.achievements && fetchedProgram.achievements.length > 0) {
            yield programAchievement_repository_1.ProgramAchivementRepository.remove(fetchedProgram.achievements);
        }
        yield program_repository_1.ProgramRepository.remove(fetchedProgram);
        return (0, http_response_1.Success)(res, {
            message: "Program removed successfully",
            data: null,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "deleteOne-program-base-error";
        error.message = error.message ? error.message : "deleteOne-program-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
exports.programController = {
    createOne,
    updateOne,
    getAll,
    getOne,
    deleteOne,
};
//# sourceMappingURL=program.controller.js.map