"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.storyController = void 0;
const badRequest_error_1 = require("../errors/badRequest.error");
const http_response_1 = require("../responses/http.response");
const path_1 = __importDefault(require("path"));
const notFound_error_1 = require("../errors/notFound.error");
const logger_1 = require("../loggers/logger");
const story_dao_1 = require("../dao/story.dao");
const story_utility_1 = require("../utilities/story.utility");
const story_dto_1 = require("../dto/story.dto");
const story_repository_1 = require("../repositories/story.repository");
const storyDao = new story_dao_1.StoryDAO();
const createOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const createRequest = res.locals.reqBody;
        if (!req.file) {
            throw new badRequest_error_1.BadRequestError("image-not-present", "Please provide required story image", 3013);
        }
        createRequest.current_user = res.locals.user;
        createRequest.image = path_1.default.join("/uploads/", req.file.filename);
        const storyEntityObj = yield (0, story_utility_1.generateStoryEntityObject)(createRequest);
        const createdStory = yield storyDao.save(storyEntityObj);
        const storyDTO = new story_dto_1.StoryDTO(createdStory);
        return (0, http_response_1.Success)(res, {
            message: "Successfully created impact story",
            data: storyDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "createOne-story-base-error";
        error.message = error.message ? error.message : "createOne-story-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const getAll = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { page, pageSize, categoryId, location } = req.query;
        logger_1.Logger.debug("query-param %s", req.query);
        if (!page || !pageSize || (pageSize && pageSize > 100)) {
            throw new notFound_error_1.NotFoundError("story-search-pageSize-more-than-100", "story search page or page size not given or page size is more than 100", 3087);
        }
        const queryBuilder = story_repository_1.StoryRepository.createQueryBuilder("story")
            .innerJoinAndSelect("story.category", "category")
            .innerJoinAndSelect("story.createdBy", "createdBy");
        if (categoryId) {
            queryBuilder.where("category.id = :categoryId", { categoryId });
        }
        if (location) {
            queryBuilder.andWhere("story.location ILIKE  :location", { location });
        }
        queryBuilder.orderBy("story.updated_at", "DESC");
        const pageNumber = page || 1;
        const itemsPerPage = pageSize || 10;
        const skip = (pageNumber - 1) * itemsPerPage;
        const totalCount = yield queryBuilder.getCount();
        queryBuilder.skip(skip).take(pageSize);
        const storyData = yield queryBuilder.getMany();
        const storyList = storyData.map(story => new story_dto_1.StoryDTO(story));
        return (0, http_response_1.Success)(res, {
            message: "Stories fetched",
            data: {
                storyList,
                total_count: totalCount,
            },
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getAll-story-base-error";
        error.message = error.message ? error.message : "getAll-story-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const getOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { storyId } = req.params;
        const fetchedStory = yield storyDao.findOneById(storyId);
        if (!fetchedStory) {
            throw new notFound_error_1.NotFoundError("getOne-no-story-with-provided-id", "No story found or the story id is invalid", 3015);
        }
        const storyDTO = new story_dto_1.StoryDTO(fetchedStory);
        return (0, http_response_1.Success)(res, {
            message: "Story detail fetched",
            data: storyDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getOne-story-base-error";
        error.message = error.message ? error.message : "getOne-story-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const updateOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const updateStoryRequestBody = res.locals.reqBody;
        const { storyId } = req.params;
        const targetStory = yield storyDao.findOneById(storyId);
        if (!targetStory) {
            throw new notFound_error_1.NotFoundError("updateOne-no-story-with-provided-id", "No story found or the story id is invalid", 3016);
        }
        req.file
            ? (updateStoryRequestBody.image = path_1.default.join("/uploads/", req.file.filename))
            : (updateStoryRequestBody.image = targetStory.image);
        const updatedStoryEntityObj = yield (0, story_utility_1.generateUpdatedStoryEntityObject)(updateStoryRequestBody, targetStory);
        const updatedStory = yield storyDao.save(updatedStoryEntityObj);
        const storyDTO = new story_dto_1.StoryDTO(updatedStory);
        return (0, http_response_1.Success)(res, {
            message: "Story details updated",
            data: storyDTO,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "getOne-story-base-error";
        error.message = error.message ? error.message : "getOne-story-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
const deleteOne = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { storyId } = req.params;
        const fetchedStory = yield storyDao.findOneById(storyId);
        if (!fetchedStory) {
            throw new notFound_error_1.NotFoundError("deleteOne-no-story-with-provided-id", "No story found or the story id is invalid", 3016);
        }
        yield story_repository_1.StoryRepository.remove(fetchedStory);
        return (0, http_response_1.Success)(res, {
            message: "Story removed successfully",
            data: null,
        });
    }
    catch (error) {
        error.origin = error.origin ? error.origin : "deleteOne-story-base-error";
        error.message = error.message ? error.message : "deleteOne-story-error";
        error.code = error.code ? error.code : 3012;
        next(error);
    }
});
exports.storyController = {
    getAll,
    createOne,
    updateOne,
    getOne,
    deleteOne,
};
//# sourceMappingURL=story.controller.js.map