"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateUserEntityObject = exports.generateRandomValidPassword = exports.isValidPassword = exports.sanitizeLoginTokenResponse = void 0;
const user_entity_1 = require("../entities/user.entity");
const sanitizeLoginTokenResponse = (tokenResponse) => {
    return {
        access: tokenResponse.data.access_token,
        refresh: tokenResponse.data.refresh_token,
    };
};
exports.sanitizeLoginTokenResponse = sanitizeLoginTokenResponse;
/**
 * * Validate the new password asynchronously:
 * * The password must meet the following criteria:
 * * - At least 8 characters long
 * * - Contains at least one lowercase letter
 * * - Contains at least one uppercase letter
 * * - Contains at least one special character (e.g., !@#$%^&*)
 * * - Contains at least one number
 * @function isValidPassword
 * @param {string} password - Password to be validated
 * @returns {Promise<boolean>} - Promise that resolves to true if the password is valid, false otherwise
 */
const isValidPassword = (password) => {
    return new Promise(resolve => {
        const passwordRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[!@#$%^&*(),.?":{}|<>]).{8,}$/;
        resolve(passwordRegex.test(password));
    });
};
exports.isValidPassword = isValidPassword;
/**
 * * Generate a random valid password asynchronously
 * * The password will follow the rule:
 * * at least 8 characters long, one capital letter, one small letter, one special character, one number
 * @function generateRandomValidPassword
 * @returns {Promise<string>} - Randomly generated valid password
 */
const generateRandomValidPassword = () => {
    return new Promise(resolve => {
        const lowercase = "abcdefghijklmnopqrstuvwxyz";
        const uppercase = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        const numbers = "0123456789";
        const specialChars = "!@#$%^&*()_+[]{}|;:,.<>?";
        let password = "";
        password += lowercase[Math.floor(Math.random() * lowercase.length)];
        password += uppercase[Math.floor(Math.random() * uppercase.length)];
        password += numbers[Math.floor(Math.random() * numbers.length)];
        password += specialChars[Math.floor(Math.random() * specialChars.length)];
        const allChars = lowercase + uppercase + numbers + specialChars;
        /**
         * * Ensure the password is at least 8 characters long by adding random characters
         */
        while (password.length < 8) {
            password += allChars[Math.floor(Math.random() * allChars.length)];
        }
        resolve(password
            .split("")
            .sort(() => 0.5 - Math.random())
            .join("")); // shuffle to avoid predictable pattern
    });
};
exports.generateRandomValidPassword = generateRandomValidPassword;
const generateUserEntityObject = (email, password, username, tempPass) => {
    const user = new user_entity_1.User();
    user.email = email;
    user.password = password;
    user.username = username;
    user.temp_pass = tempPass;
    return user;
};
exports.generateUserEntityObject = generateUserEntityObject;
//# sourceMappingURL=user.utility.js.map